define([
    'underscore',
    'modules/appointments/booked/booked-appointments-collection',
], function(_, Collection) {
    'use strict';

    describe('Booked Appointments', function() {
        var Model;

        beforeEach(function() {
            Model = Collection.prototype.model;
        });

        describe('Model', function() {

            var data;
            var options;
            var model;

            beforeEach(function() {
                data = {
                    currentStatus: 'testing',
                    clinic: {friendlyName: 'FriendlyName'},
                };
                options = {
                    parse: true,
                    siteCode: 100,
                    facilityName: 'NSA',
                };

                model = new Model(data, options);
            });


            it('parses data correctly', function() {
                expect(model.get('siteCode')).toBe(options.siteCode);
                expect(model.get('facilityName')).toBe(data.clinic.friendlyName);
                expect(model.get('currentStatus')).toBe(data.currentStatus);
            });

            it('evaluates isCancelled correctly', function() {
                var statusList = [
                    'NO-SHOW',
                    'CANCELLED BY CLINIC',
                    'NO-SHOW & AUTO RE-BOOK',
                    'CANCELLED BY CLINIC & AUTO RE-BOOK',
                    'INPATIENT APPOINTMENT',
                    'CANCELLED BY PATIENT',
                    'CANCELLED BY PATIENT & AUTO-REBOOK',
                    'NO ACTION TAKEN',
                ];

                expect(model.isCancelled()).toBe(false);
                _.each(statusList, function(status) {
                    model.set('currentStatus', status);
                    expect(model.isCancelled()).toBe(true);
                });
            });
        });

        describe('Collection', function() {
            var data;
            var collection;

            beforeEach(function() {
                data = [
                    {shouldExist: true, appointmentTime: '2019-10-13T19:32:46.505Z', currentStatus: 'test', clinic: {friendlyName: 'FriendlyName'}, siteCode: 100}, // HIGHEST
                    {shouldExist: true, appointmentTime: '2025-10-13T19:32:46.505Z', currentStatus: 'test', clinic: {friendlyName: 'FriendlyName'}, siteCode: 100}, // LOWEST
                    {shouldExist: true, appointmentTime: '2020-10-13T19:32:46.505Z', currentStatus: 'test', clinic: {friendlyName: 'FriendlyName'}, siteCode: 100}, // MIDDLE
                    {shouldExist: false, appointmentTime: 'wrong value', currentStatus: 'test', clinic: {friendlyName: 'FriendlyName'}, siteCode: 100}, // Bad date string
                    {shouldExist: false, appointmentTime: '2016-10-13T19:32:46.505Z', currentStatus: 'NO-SHOW', clinic: {friendlyName: 'FriendlyName'}, siteCode: 100}, // Canceled already
                ];

                collection = new Collection({bookedAppointmentCollections: [{bookedAppointments: data, collectionName: 'bookedAppointments'}]}, {parse: true});
            });

            it('parses the bad data out of the collection', function() {
                expect(collection.length).toBe(3);
                collection.each(function(model) {
                    expect(model.get('shouldExist')).toBe(true);
                });
            });
        });

        describe('Booked Appointments Collection', function() {
            var collection;
            var validData;
            var filteredData;
            var timeFixedData;
            beforeEach(function() {
                validData = [
                    {shouldExist: true, appointmentTime: '2025-10-13T19:32:46.505Z', currentStatus: 'test', clinic: {friendlyName: 'FriendlyName'}, siteCode: 100}, // HIGHEST
                    {shouldExist: true, appointmentTime: '2029-10-13T19:32:46.505Z', currentStatus: 'test', clinic: {friendlyName: 'FriendlyName'}, siteCode: 100}, // LOWEST

                ];
                filteredData = [
                    {shouldExist: false, appointmentTime: '2017-10-13T19:32:46.505Z', currentStatus: 'CANCELLED BY CLINIC', clinic: {friendlyName: 'FriendlyName'}, siteCode: 100}, // HIGHEST
                    {shouldExist: false, appointmentTime: 'Bad Date', timeZone: '07:00 PDT', currentStatus: 'test', clinic: {friendlyName: 'FriendlyName'}, siteCode: 100},
                ];
                timeFixedData = [
                    {shouldExist: true, appointmentTime: '03/31/2018 19:00:00', timeZone: '-07:00 PDT', currentStatus: 'test', clinic: {friendlyName: 'FriendlyName'}, siteCode: 100},
                    {shouldExist: true, appointmentTime: '03/31/2018 19:00:00', timeZone: '07:00 PDT', currentStatus: 'test', clinic: {friendlyName: 'FriendlyName'}, siteCode: 100},
                ];

                collection = new Collection({bookedAppointmentCollections: []}, {parse: false});
            });
            describe('Filter', function() {
                it('removes items with cancelled status or invalid time', function () {
                    _.each(validData, function(appointment) {
                        expect(collection.filterItem(appointment)).toEqual(true);
                    });
                    _.each(filteredData, function(appointment) {
                        expect(collection.filterItem(appointment)).toEqual(false);
                    });
                });
            });
            describe('FixTime', function() {
                it('doesn\'t update the appointmentTime if timezone does not exist', function() {
                    _.each(validData, function (appointment) {
                        var time = appointment.appointmentTime;
                        collection.fixTime(appointment);
                        expect(appointment.appointmentTime).toEqual(time);
                    });
                });
                it('updates the appointmentTime if timezone exists', function() {
                    _.each(timeFixedData, function(appointment) {
                        var time = appointment.appointmentTime;
                        collection.fixTime(appointment);
                        expect(appointment.appointmentTime).not.toEqual(time);
                    });
                });
                it('properly updates the hours based on the timezone passed', function() {
                    _.each(timeFixedData, function(appointment) {
                        collection.fixTime(appointment);
                    });
                    expect(timeFixedData[0].appointmentTime).toEqual('03/31/2018 12:00:00');
                    expect(timeFixedData[1].appointmentTime).toEqual('04/01/2018 02:00:00');
                });
            });
        });


    });
});

